package bin.mt.plugin.api.regex;

import java.util.regex.PatternSyntaxException;

import bin.mt.plugin.api.editor.BufferedText;
import bin.mt.plugin.api.util.ServiceLoader;

/**
 * MT 正则表达式工具类，提供正则表达式模式编译和替换模板验证的静态方法
 * <p>
 * MT 正则表达式库相比系统库提供了更多的功能，例如支持全词匹配、任意位置 lookingAt、支持快速生成匹配结果快照等。
 * 在性能上也针对性进行了优化，对于传入的类型匹配文本，直接在 CharSequence 接口方法上进行匹配，而非系统库那样调用
 * toString() 方法（这对文本编辑器来说是个性能灾难），在一些极端情况下也能有效避免匹配卡死问题。
 *
 * @see BufferedText#matcher(String)
 * @see BufferedText#matcher(Pattern)
 */
public final class Regex {

    /**
     * 编译正则表达式模式
     *
     * @param pattern 正则表达式字符串
     * @return 编译后的Pattern对象
     * @throws PatternSyntaxException 当正则表达式语法错误时抛出
     */
    public static Pattern compile(String pattern) throws PatternSyntaxException {
        return ServiceLoader.get(Impl.class).compileImpl(pattern);
    }

    /**
     * 使用指定标志编译正则表达式模式
     *
     * @param pattern 正则表达式字符串
     * @param flags   匹配标志的组合值
     * @return 编译后的Pattern对象
     * @throws PatternSyntaxException 当正则表达式语法错误时抛出
     */
    public static Pattern compile(String pattern, int flags) throws PatternSyntaxException {
        return ServiceLoader.get(Impl.class).compileImpl(pattern, flags);
    }

    /**
     * 检查替换模板的语法正确性
     *
     * <p> 用于验证替换字符串中的捕获组引用格式是否正确
     *
     * @param replacementTemplate 替换模板字符串
     * @throws PatternSyntaxException 当替换模板语法错误时抛出
     */
    public static void checkReplacementTemplate(String replacementTemplate) throws PatternSyntaxException {
        ServiceLoader.get(Impl.class).checkReplacementTemplateImpl(replacementTemplate);
    }

    private Regex() {
    }

    interface Impl {
        Pattern compileImpl(String pattern);

        Pattern compileImpl(String pattern, int flags);

        void checkReplacementTemplateImpl(String replacementTemplate);
    }
}
