package bin.mt.plugin.api.regex;

/**
 * 正则表达式模式接口
 * 定义了正则表达式的各种匹配标志位和基本操作方法
 */
public interface Pattern {

    /**
     * Unix行模式标志 - 只识别'\n'作为行终止符
     *
     * <p> 可通过嵌入标志(?d)启用
     */
    int UNIX_LINES = 0x01;

    /**
     * 忽略大小写匹配标志
     *
     * <p> 默认仅对US-ASCII字符生效，配合UNICODE_CASE可支持Unicode字符，
     * <p> 可通过嵌入标志(?i)启用，可能影响性能
     */
    int CASE_INSENSITIVE = 0x02;

    /**
     * 注释模式标志 - 忽略空白字符和#开头的注释
     *
     * <p> 可通过嵌入标志(?x)启用
     */
    int COMMENTS = 0x04;

    /**
     * 多行模式标志 - ^和$匹配行的开始和结束
     *
     * <p> 默认情况下^和$只匹配整个输入序列的开始和结束，
     * <p> 可通过嵌入标志(?m)启用
     */
    int MULTILINE = 0x08;

    /**
     * 字面量解析标志 - 将模式字符串视为普通字符
     *
     * <p> 元字符和转义序列不具有特殊含义，
     * <p> 与CASE_INSENSITIVE和UNICODE_CASE标志可同时使用
     */
    int LITERAL = 0x10;

    /**
     * 单行模式标志 - 使'.'匹配包括换行符在内的任意字符
     *
     * <p> 可通过嵌入标志(?s)启用
     */
    int DOTALL = 0x20;

    /**
     * Unicode大小写折叠标志
     *
     * <p> 与CASE_INSENSITIVE配合使用时按Unicode标准进行大小写匹配，
     * <p> 可通过嵌入标志(?u)启用，可能影响性能
     */
    int UNICODE_CASE = 0x40;

    /**
     * 规范等价标志 - 按Unicode规范分解进行字符匹配
     *
     * <p> 例如"a\u030A"可匹配"\u00E5"，
     * <p> 可能影响性能
     */
    int CANON_EQ = 0x80;

    /**
     * Unicode字符类标志 - 启用Unicode版本的预定义字符类和POSIX字符类
     *
     * <p> 符合Unicode技术标准#18规范，
     * <p> 可通过嵌入标志(?U)启用，隐含启用UNICODE_CASE，可能影响性能
     */
    int UNICODE_CHARACTER_CLASS = 0x100;

    /**
     * 全词匹配标志 - 匹配完整单词
     *
     * <p> MT正则自定义扩展标志，用于精确的单词边界匹配
     */
    int MATCH_WHOLE_WORD = 0x200;

    /**
     * 获取正则表达式模式字符串
     *
     * @return 模式字符串
     */
    String pattern();

    /**
     * 获取正则表达式标志位
     *
     * @return 标志位的组合值
     */
    int flags();

    /**
     * 创建用于匹配指定文本的匹配器
     *
     * @param text 要匹配的文本序列
     * @return 新创建的匹配器实例
     */
    Matcher matcher(CharSequence text);
}
