package bin.mt.plugin.api.regex;

/**
 * 正则表达式匹配器接口
 * 提供正则表达式匹配和查找功能
 */
public interface Matcher {

    /**
     * 获取当前匹配器关联的模式
     *
     * @return 当前匹配器的模式
     */
    Pattern pattern();

    /**
     * 获取当前匹配器关联的文本序列
     *
     * @return 当前匹配器的文本序列
     */
    CharSequence getText();

    /**
     * 重置匹配器状态
     *
     * @return 返回当前匹配器实例，支持链式调用
     */
    Matcher reset();

    /**
     * 重置匹配器并设置新的输入文本
     *
     * @param text 新的输入文本序列
     * @return 返回当前匹配器实例，支持链式调用
     */
    Matcher reset(CharSequence text);

    /**
     * 判断整个输入序列是否与正则表达式完全匹配
     *
     * @return 如果整个输入序列匹配则返回true，否则返回false
     */
    boolean matches();

    /**
     * 查找下一个匹配的子序列
     *
     * @return 如果找到匹配的子序列则返回true，否则返回false
     */
    boolean find();

    /**
     * 从指定位置开始查找下一个匹配的子序列
     *
     * @param start 开始查找的位置索引
     * @return 如果找到匹配的子序列则返回true，否则返回false
     */
    boolean find(int start);

    /**
     * 判断输入序列的开头是否与正则表达式匹配
     *
     * @return 如果输入序列的开头匹配则返回true，否则返回false
     */
    boolean lookingAt();

    /**
     * 从指定位置开始判断输入序列是否与正则表达式匹配
     *
     * @param start 开始匹配的位置索引
     * @return 如果从指定位置开始匹配则返回true，否则返回false
     */
    boolean lookingAt(int start);

    /**
     * 创建当前匹配状态的快照
     * <p>
     * 用于保存当前匹配结果，便于后续访问或恢复
     *
     * @return 匹配状态快照对象
     */
    MatcherSnapshot toSnapshot();

    /**
     * 获取上一次匹配操作中匹配子序列的起始位置
     *
     * @return 匹配子序列的起始索引
     */
    int start();

    /**
     * 获取上一次匹配操作中指定捕获组的起始位置
     *
     * @param group 捕获组的索引（0表示整个匹配）
     * @return 指定捕获组的起始索引
     */
    int start(int group);

    /**
     * 获取上一次匹配操作中匹配子序列的结束位置
     *
     * @return 匹配子序列的结束索引（不包含该位置的字符）
     */
    int end();

    /**
     * 获取上一次匹配操作中指定捕获组的结束位置
     *
     * @param group 捕获组的索引（0表示整个匹配）
     * @return 指定捕获组的结束索引（不包含该位置的字符）
     */
    int end(int group);

    /**
     * 获取上一次匹配操作中匹配的子序列内容
     *
     * @return 匹配的字符串内容
     */
    String group();

    /**
     * 获取上一次匹配操作中指定捕获组的内容
     *
     * @param group 捕获组的索引（0表示整个匹配，1及以上表示各个捕获组）
     * @return 指定捕获组的字符串内容
     */
    String group(int group);

    /**
     * 获取正则表达式中捕获组的数量
     *
     * @return 捕获组的数量（不包括组0，即整个匹配）
     */
    int groupCount();

    /**
     * 设置匹配器的匹配区域
     * <p>
     * 限制匹配操作只在指定的字符范围内进行
     *
     * @param start 区域起始位置（包含）
     * @param end   区域结束位置（不包含）
     * @return 返回当前匹配器实例，支持链式调用
     */
    Matcher region(int start, int end);

    /**
     * 获取当前匹配区域的起始位置
     *
     * @return 匹配区域的起始索引
     */
    int regionStart();

    /**
     * 获取当前匹配区域的结束位置
     *
     * @return 匹配区域的结束索引
     */
    int regionEnd();

    /**
     * 检查是否启用了透明边界模式
     *
     * @return 如果启用透明边界则返回true，否则返回false
     * @see #useTransparentBounds(boolean)
     */
    boolean hasTransparentBounds();

    /**
     * 设置此匹配器区域边界的透明性
     *
     * <p> 透明边界：前瞻、后顾和边界匹配构造可以看到区域边界之外的内容
     * <p> 不透明边界：这些构造无法越过区域边界查看外部内容
     * <p> 默认使用不透明边界
     *
     * @param b true使用透明边界，false使用不透明边界
     * @return 此匹配器
     */
    Matcher useTransparentBounds(boolean b);

    /**
     * 检查是否启用了锚定边界模式
     * <p>
     * 锚定边界模式下，区域边界被视为输入序列的边界
     *
     * @return 如果启用锚定边界则返回true，否则返回false
     */
    boolean hasAnchoringBounds();

    /**
     * 设置锚定边界模式
     * <p>
     * 控制区域边界是否被视为输入序列的边界
     *
     * @param b true表示启用锚定边界，false表示禁用
     * @return 返回当前匹配器实例，支持链式调用
     */
    Matcher useAnchoringBounds(boolean b);

    /**
     * 根据替换模式获取替换后的文本
     * <p>
     * 将匹配的内容按照指定的替换模式进行替换
     *
     * @param replacementTemplate 替换模式字符串，可包含捕获组引用（如$1, $2等）
     * @return 替换后的文本内容
     */
    String getReplacedText(String replacementTemplate);

    /**
     * 将替换结果追加到StringBuilder中
     * <p>
     * 执行替换操作并将结果追加到指定的StringBuilder对象
     *
     * @param sb                  目标StringBuilder对象
     * @param replacementTemplate 替换模式字符串，可包含捕获组引用（如$1, $2等）
     */
    void appendReplacement(StringBuilder sb, String replacementTemplate);

}
