package bin.mt.plugin.api.editor;

import android.graphics.drawable.Drawable;

import androidx.annotation.NonNull;

import bin.mt.plugin.api.PluginContext;
import bin.mt.plugin.api.ui.PluginUI;

/**
 * 文本编辑器浮动菜单扩展接口
 */
public interface TextEditorFloatingMenu {

    /**
     * 初始化方法，在菜单实例创建后立即调用
     * <p>
     * 此方法在每个实例的生命周期中仅调用一次，用于进行必要的初始化工作，
     * 如资源加载、配置读取、状态初始化等。
     *
     * @param context MT插件上下文，提供插件运行所需的环境和资源访问能力
     */
    void init(PluginContext context);

    /**
     * 获取插件上下文
     * <p>
     * 返回在init()方法中传入的PluginContext实例，用于后续操作中
     * 访问插件框架提供的各种服务和资源。
     *
     * @return 插件上下文实例，不会为null
     */
    @NonNull
    PluginContext getContext();

    /**
     * 判断当前菜单是否启用
     * <p>
     * 返回 {@code false} 时，用户即使在设置界面也无法看到此菜单
     * <p>
     * 典型使用场景：
     * <ul>
     * <li><b>功能开关控制</b>：插件提供设置界面让用户选择启用或禁用某些菜单功能</li>
     * <li><b>条件性启用</b>：根据运行环境、权限状态或其他条件动态决定菜单可用性</li>
     * <li><b>精简菜单列表</b>：当插件实现多个菜单接口时，避免选项过多造成用户困扰</li>
     * </ul>
     * <p>
     * <b>注意</b>：此方法可能被频繁调用，建议避免在其中执行耗时操作。
     *
     * @return {@code true} 表示菜单启用，{@code false} 表示菜单禁用
     */
    boolean isEnabled();

    /**
     * 获取菜单的显示名称，菜单命名请尽可能简短
     * <p>
     * 如果返回的名称为{key}格式，将会自动调用
     * getContext().getLocalString().getOrDefault("key", "{key}") 尝试转化为本地化文本
     *
     * @return 菜单名称，不能为null或空字符串
     */
    @NonNull
    String name();

    /**
     * 获取菜单的图标
     *
     * @return 菜单图标，不能为null
     */
    @NonNull
    Drawable icon();

    /**
     * 插件按钮点击时的回调方法
     * <p>
     * 在浮动菜单编辑界面，每个插件功能项的右边有一个插件图标样式的按钮，当该按钮被点击后会调用此方法。
     *
     * @param pluginUI 插件UI接口
     */
    void onPluginButtonClick(@NonNull PluginUI pluginUI);

    /**
     * 测试是否应该显示菜单
     * <p>
     * 该方法在弹出菜单前调用，您需要在此方法中对文本编辑器状态进行检查，以决定是否显示菜单。
     * 例如实现一个处理选中文本的功能，那么需要检查当前文本编辑器是否处于多选状态。
     * <p>
     * <b>注意</b>：此方法可能被频繁调用，建议避免在其中执行耗时操作。
     *
     * @param editor 文本编辑器实例
     * @return {@code true} 表示菜单显示，{@code false} 表示菜单隐藏
     */
    boolean checkVisible(@NonNull TextEditor editor);

    /**
     * 菜单功能的核心方法
     * <p>
     * 当用户点击菜单时调用此方法，实现具体的功能逻辑，可以通过editor参数操作文本内容。
     *
     * @param editor 文本编辑器实例，提供文本操作接口
     */
    void onMenuClick(@NonNull PluginUI pluginUI, @NonNull TextEditor editor);

}
