package bin.mt.plugin.api.util;

import java.util.concurrent.Executor;

import bin.mt.plugin.api.PluginContext;

/**
 * 异步任务执行器
 * <p>
 * 提供一个简化的异步任务执行框架，支持在后台线程执行耗时操作，
 * 并在主线程中处理结果和异常。该类封装了线程切换的复杂性，
 * 提供了清晰的生命周期回调方法。
 * <p>
 * 执行流程：
 * 1. beforeThread() - 主线程预处理
 * 2. onThread() - 后台线程执行主要任务
 * 3. afterThread() - 主线程后处理
 * 4. onFinally() - 最终清理工作（无论成功失败都会执行）
 * <p>
 * 异常处理：
 * - 任何阶段发生异常都会调用 onException()
 * - 特别处理 OutOfMemoryError，转换为带本地化消息的异常
 * - 所有异常处理都在主线程中进行
 */
public abstract class AsyncTask {

    /**
     * 线程池执行器，用于执行后台任务
     */
    private Executor executor = android.os.AsyncTask.THREAD_POOL_EXECUTOR;

    /**
     * 插件上下文，用于获取资源和字符串
     */
    protected final PluginContext context;

    /**
     * 构造函数
     *
     * @param context 插件上下文，不能为null
     */
    public AsyncTask(PluginContext context) {
        this.context = context;
    }

    /**
     * 任务开始前的预处理方法
     * <p>
     * 在主线程中执行，通常用于：
     * - 显示加载进度条
     * - 初始化UI状态
     *
     * @throws Exception 如果预处理失败，整个任务将被取消
     */
    protected abstract void beforeThread() throws Exception;

    /**
     * 后台线程执行的主要任务
     * <p>
     * 在后台线程中执行，适合进行：
     * - 网络请求
     * - 文件IO操作
     * - 复杂计算
     * <p>
     * 注意：不要在此方法中直接操作UI
     *
     * @throws Exception 任务执行过程中的异常
     */
    protected abstract void onThread() throws Exception;

    /**
     * 任务完成后的处理方法
     * <p>
     * 在主线程中执行，通常用于：
     * - 更新UI显示结果
     * - 隐藏加载进度条
     * - 处理任务执行结果
     *
     * @throws Exception 后处理过程中的异常
     */
    protected abstract void afterThread() throws Exception;

    /**
     * 异常处理方法
     * <p>
     * 在主线程中执行，用于处理任务执行过程中发生的所有异常。
     * 包括 beforeThread、onThread、afterThread 中抛出的异常。
     *
     * @param e 捕获到的异常对象
     */
    protected abstract void onException(Exception e);

    /**
     * 最终清理方法
     * <p>
     * 在主线程中执行，无论任务成功、失败或被取消都会调用。
     * 通常用于：
     * - 释放资源
     * - 重置状态
     * - 清理临时数据
     * <p>
     * 注意：此方法不应抛出异常
     */
    protected abstract void onFinally();

    /**
     * 启动异步任务
     * <p>
     * 按照以下顺序执行：
     * 1. 在主线程执行 beforeThread()
     * 2. 在后台线程执行 onThread()
     * 3. 在主线程执行 afterThread()
     * 4. 最后在主线程执行 onFinally()
     * <p>
     * 如果任何步骤发生异常，会调用 onException() 处理，
     * 并确保 onFinally() 始终被执行。
     */
    public void start() {
        try {
            // 主线程预处理
            beforeThread();
        } catch (Exception e) {
            // 预处理失败，直接处理异常并结束
            try {
                onException(e);
            } finally {
                onFinally();
            }
            return;
        }

        // 创建后台任务
        Runnable runnable = () -> {
            try {
                // 执行后台任务
                onThread();
            } catch (Exception e) {
                // 处理普通异常
                handleError(e);
                return;
            } catch (OutOfMemoryError e) {
                // 处理内存溢出错误，转换为带本地化消息的异常
                handleError(new Exception(context.getString("{out_of_memory_error}"), e));
                return;
            } catch (Throwable e) {
                // 处理其他所有错误
                handleError(new Exception(e));
                return;
            }

            // 后台任务成功完成，切换到主线程执行后处理
            ThreadUtil.post(() -> {
                try {
                    // 主线程后处理
                    afterThread();
                } catch (Exception e) {
                    // 后处理异常
                    onException(e);
                } catch (OutOfMemoryError e) {
                    // 后处理内存溢出
                    onException(new Exception(context.getString("{out_of_memory_error}"), e));
                } catch (Throwable e) {
                    // 后处理其他错误
                    onException(new Exception(e));
                } finally {
                    // 确保最终清理方法被执行
                    onFinally();
                }
            });
        };

        // 提交任务到线程池执行
        executor.execute(runnable);
    }

    /**
     * 统一的错误处理方法
     * <p>
     * 将异常处理切换到主线程执行，确保异常回调和清理工作都在主线程中进行，避免线程安全问题。
     *
     * @param e 需要处理的异常
     */
    private void handleError(Exception e) {
        ThreadUtil.post(() -> {
            try {
                // 在主线程中处理异常
                onException(e);
            } finally {
                // 确保清理工作被执行
                onFinally();
            }
        });
    }
}
