package bin.mt.plugin.api.ui.dialog;

import android.view.KeyEvent;

import androidx.annotation.Nullable;

import bin.mt.plugin.api.ui.PluginButton;
import bin.mt.plugin.api.ui.PluginUI;
import bin.mt.plugin.api.ui.PluginView;
import bin.mt.plugin.api.ui.builder.PluginBaseTextViewBuilder;

/**
 * 插件对话框接口
 * <p>
 * 提供对话框的核心功能，包括显示、隐藏、按钮管理和各种事件监听，类似于Android原生的Dialog
 * <p>
 * 对话框的三种按钮在点击后，对话框会自动消失并调用事件监听器，如果不希望对话框消失，可以使用以下方法：
 * <pre>{@code
 * PluginDialog dialog = dialogBuilder
 *     .setTitle("Test")
 *     .setPositiveButton("OK", null)
 *     .show(); // 必须在show之后才能获取到按钮
 *
 * dialog.getPositiveButton().setOnClickListener(v -> {
 *    // 处理点击事件
 *    // 如果要让对话框消失需手动调用 dialog.dismiss();
 * });
 * }</pre>
 */
public interface PluginDialog {

    /**
     * 积极按钮标识符（通常是"确定"按钮）
     */
    int BUTTON_POSITIVE = -1;

    /**
     * 消极按钮标识符（通常是"取消"按钮）
     */
    int BUTTON_NEGATIVE = -2;

    /**
     * 中性按钮标识符（通常是"稍后"等中性选项按钮）
     */
    int BUTTON_NEUTRAL = -3;

    /**
     * 获取关联的插件UI实例
     *
     * @return 插件UI实例，用于访问插件的UI上下文
     */
    PluginUI getPluginUI();

    /**
     * 显示对话框
     * 如果对话框已经显示，则此方法无效果
     */
    void show();

    /**
     * 关闭对话框
     * 会触发OnDismissListener回调
     */
    void dismiss();

    /**
     * 取消对话框
     * 会触发OnCancelListener和OnDismissListener回调
     */
    void cancel();

    /**
     * 获取积极按钮实例
     * <p>
     * 注意：如果通过builder.create()创建了此对话框，那么必须至少调用一次dialog.show()
     * 再获取按钮实例，否则将返回null
     *
     * @return 积极按钮对象，可能为null如果未设置
     */
    PluginButton getPositiveButton();

    /**
     * 获取消极按钮实例
     * <p>
     * 注意：如果通过builder.create()创建了此对话框，那么必须至少调用一次dialog.show()
     * 再获取按钮实例，否则将返回null
     *
     * @return 消极按钮对象，可能为null如果未设置
     */
    PluginButton getNegativeButton();

    /**
     * 获取中性按钮实例
     * <p>
     * 注意：如果通过builder.create()创建了此对话框，那么必须至少调用一次dialog.show()
     * 再获取按钮实例，否则将返回null
     *
     * @return 中性按钮对象，可能为null如果未设置
     */
    PluginButton getNeutralButton();

    /**
     * 动态设置积极按钮
     *
     * @param text     按钮文本
     * @param listener 点击事件监听器
     */
    void setPositiveButton(CharSequence text, OnClickListener listener);

    /**
     * 动态设置消极按钮
     *
     * @param text     按钮文本
     * @param listener 点击事件监听器
     */
    void setNegativeButton(CharSequence text, OnClickListener listener);

    /**
     * 动态设置中性按钮
     *
     * @param text     按钮文本
     * @param listener 点击事件监听器
     */
    void setNeutralButton(CharSequence text, OnClickListener listener);

    /**
     * 检查对话框是否正在显示
     *
     * @return true表示正在显示，false表示已关闭或未显示
     */
    boolean isShowing();

    /**
     * 设置自定义视图
     *
     * @param view 自定义视图，将替换对话框的内容区域
     */
    void setView(PluginView view);

    /**
     * 设置对话框标题
     *
     * @param title 标题文本
     */
    void setTitle(CharSequence title);

    /**
     * 设置对话框消息内容
     *
     * @param title 消息文本（参数名应该是message，这里可能是命名错误）
     */
    void setMessage(CharSequence title);

    /**
     * 设置对话框是否可取消
     *
     * @param cancelable true表示可以通过返回键或点击外部区域取消
     */
    void setCancelable(boolean cancelable);

    /**
     * 设置对话框显示事件监听器
     *
     * @param listener 显示事件监听器，可为null
     */
    void setOnShowListener(@Nullable OnShowListener listener);

    /**
     * 设置对话框消失事件监听器
     *
     * @param listener 消失事件监听器，可为null
     */
    void setOnDismissListener(@Nullable OnDismissListener listener);

    /**
     * 设置对话框取消事件监听器
     *
     * @param listener 取消事件监听器，可为null
     */
    void setOnCancelListener(@Nullable OnCancelListener listener);

    /**
     * 设置按键事件监听器
     *
     * @param listener 按键事件监听器，可为null
     */
    void setOnKeyListener(@Nullable OnKeyListener listener);

    /**
     * 对话框取消事件监听器接口
     * 当用户通过返回键或点击外部区域取消对话框时触发
     */
    interface OnCancelListener {
        /**
         * 对话框被取消时的回调方法
         *
         * @param dialog 被取消的对话框实例
         */
        void onCancel(PluginDialog dialog);
    }

    /**
     * 对话框消失事件监听器接口
     * 当对话框因任何原因消失时触发（包括取消、按钮点击等）
     */
    interface OnDismissListener {
        /**
         * 对话框消失时的回调方法
         *
         * @param dialog 消失的对话框实例
         */
        void onDismiss(PluginDialog dialog);
    }

    /**
     * 对话框显示事件监听器接口
     * 当对话框显示时触发
     */
    interface OnShowListener {
        /**
         * 对话框显示时的回调方法
         *
         * @param dialog 显示的对话框实例
         */
        void onShow(PluginDialog dialog);
    }

    /**
     * 对话框按钮点击事件监听器接口
     * 用于处理对话框按钮和列表项的点击事件
     */
    interface OnClickListener {
        /**
         * 按钮或列表项被点击时的回调方法
         *
         * @param dialog 触发事件的对话框实例
         * @param which  被点击的按钮标识符或列表项索引
         *               对于按钮：BUTTON_POSITIVE、BUTTON_NEGATIVE、BUTTON_NEUTRAL
         *               对于列表项：对应的数组索引
         */
        void onClick(PluginDialog dialog, int which);
    }

    /**
     * 多选列表项点击事件监听器接口
     * 专门用于处理多选列表的选择状态变化
     */
    interface OnMultiChoiceClickListener {
        /**
         * 多选列表项状态改变时的回调方法
         *
         * @param dialog    触发事件的对话框实例
         * @param which     被点击的列表项索引
         * @param isChecked 该列表项的新选中状态
         */
        void onClick(PluginDialog dialog, int which, boolean isChecked);
    }

    /**
     * 按键事件监听器接口
     * 用于处理对话框的按键事件，如返回键、菜单键等
     */
    interface OnKeyListener {
        /**
         * 按键事件处理回调方法
         *
         * @param dialog  触发事件的对话框实例
         * @param keyCode 按键代码
         * @param event   按键事件对象
         * @return true表示事件已处理，false表示继续传递事件
         */
        boolean onKey(PluginDialog dialog, int keyCode, KeyEvent event);
    }

    /**
     * 插件对话框构建器接口
     */
    interface Builder {

        /**
         * 设置对话框标题
         * <p>
         * 如果title类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
         * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
         *
         * @param title 标题文本，支持CharSequence类型以便支持富文本
         * @return 返回构建器实例，支持链式调用
         * @see PluginBaseTextViewBuilder#text(CharSequence)
         */
        Builder setTitle(CharSequence title);

        /**
         * 设置对话框消息内容
         * <p>
         * 如果message类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
         * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
         *
         * @param message 消息文本，支持CharSequence类型以便支持富文本
         * @return 返回构建器实例，支持链式调用
         * @see PluginBaseTextViewBuilder#text(CharSequence)
         */
        Builder setMessage(CharSequence message);

        /**
         * 设置积极按钮（通常是"确定"按钮）
         * <p>
         * 如果text类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
         * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
         *
         * @param text     按钮显示的文本
         * @param listener 按钮点击事件监听器
         * @return 返回构建器实例，支持链式调用
         * @see PluginBaseTextViewBuilder#text(CharSequence)
         */
        Builder setPositiveButton(CharSequence text, OnClickListener listener);

        /**
         * 设置消极按钮（通常是"取消"按钮）
         * <p>
         * 如果text类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
         * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
         *
         * @param text     按钮显示的文本
         * @param listener 按钮点击事件监听器
         * @return 返回构建器实例，支持链式调用
         * @see PluginBaseTextViewBuilder#text(CharSequence)
         */
        Builder setNegativeButton(CharSequence text, OnClickListener listener);

        /**
         * 设置中性按钮（通常是"稍后"或其他中性选项）
         * <p>
         * 如果text类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
         * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
         *
         * @param text     按钮显示的文本
         * @param listener 按钮点击事件监听器
         * @return 返回构建器实例，支持链式调用
         * @see PluginBaseTextViewBuilder#text(CharSequence)
         */
        Builder setNeutralButton(CharSequence text, OnClickListener listener);

        /**
         * 设置对话框是否可以通过返回键或点击外部区域取消
         *
         * @param cancelable true表示可取消，false表示不可取消
         * @return 返回构建器实例，支持链式调用
         */
        Builder setCancelable(boolean cancelable);

        /**
         * 设置对话框显示事件监听器
         *
         * @param listener 显示事件监听器，当对话框显示时回调
         * @return 返回构建器实例，支持链式调用
         */
        Builder setOnShowListener(OnShowListener listener);

        /**
         * 设置对话框消失事件监听器
         *
         * @param listener 消失事件监听器，当对话框消失时回调（包括取消和正常关闭）
         * @return 返回构建器实例，支持链式调用
         */
        Builder setOnDismissListener(OnDismissListener listener);

        /**
         * 设置对话框取消事件监听器
         *
         * @param listener 取消事件监听器，当对话框被取消时回调
         * @return 返回构建器实例，支持链式调用
         */
        Builder setOnCancelListener(OnCancelListener listener);

        /**
         * 设置按键事件监听器
         *
         * @param listener 按键事件监听器，处理对话框的按键事件
         * @return 返回构建器实例，支持链式调用
         */
        Builder setOnKeyListener(OnKeyListener listener);

        /**
         * 设置简单列表项（单选列表）
         *
         * @param items    列表项文本数组
         * @param listener 列表项点击监听器
         * @return 返回构建器实例，支持链式调用
         */
        Builder setItems(CharSequence[] items, OnClickListener listener);

        /**
         * 设置多选列表项
         *
         * @param items        列表项文本数组
         * @param checkedItems 对应每个列表项的初始选中状态
         * @param listener     多选列表项点击监听器
         * @return 返回构建器实例，支持链式调用
         */
        Builder setMultiChoiceItems(CharSequence[] items, boolean[] checkedItems, OnMultiChoiceClickListener listener);

        /**
         * 设置单选列表项（带单选按钮的列表）
         *
         * @param items       列表项文本数组
         * @param checkedItem 初始选中项的索引，-1表示无选中项
         * @param listener    单选列表项点击监听器
         * @return 返回构建器实例，支持链式调用
         */
        Builder setSingleChoiceItems(CharSequence[] items, int checkedItem, OnClickListener listener);

        /**
         * 设置自定义视图
         *
         * @param view 自定义的插件视图，将显示在对话框内容区域
         * @return 返回构建器实例，支持链式调用
         */
        Builder setView(PluginView view);

        /**
         * 创建对话框实例
         *
         * @return 创建好的PluginDialog实例，但未显示
         */
        PluginDialog create();

        /**
         * 创建并立即显示对话框
         *
         * @return 创建好并已显示的PluginDialog实例
         */
        PluginDialog show();
    }

}
