package bin.mt.plugin.api.ui.dialog;

import android.text.TextUtils;
import android.view.Gravity;

import java.lang.ref.WeakReference;

import bin.mt.plugin.api.ui.PluginProgressBar;
import bin.mt.plugin.api.ui.PluginTextView;
import bin.mt.plugin.api.ui.PluginUI;
import bin.mt.plugin.api.ui.PluginView;
import bin.mt.plugin.api.util.ThreadUtil;
import bin.mt.plugin.api.util.UIUpdater;
import bin.mt.plugin.api.util.UIUpdaterGroup;

/**
 * 加载对话框类
 * <p>
 * 提供带有进度条和文本信息的加载提示对话框，支持延迟显示和动态更新消息
 */
public class LoadingDialog extends BaseCancelableDialog<LoadingDialog> {
    private final PluginDialog dialog;
    private final PluginTextView messageView;
    private final PluginTextView secondaryMessageView;
    private final UIUpdater<CharSequence> messageUpdater;
    private final UIUpdater<CharSequence> secondaryMessageUpdater;
    private boolean delayShow;

    public LoadingDialog(PluginUI pluginUI) {
        super(pluginUI);
        // 构建对话框布局：水平布局包含进度条和垂直文本布局
        PluginView pluginView = pluginUI.buildHorizontalLayout()
                .gravity(Gravity.CENTER_VERTICAL) // 垂直居中对齐
                .paddingDp(15) // 设置15dp内边距
                // 添加圆形进度条
                .addProgressBar().style(PluginProgressBar.Style.CIRCULAR)
                // 添加垂直文本布局
                .addVerticalLayout().widthMatchParent().marginLeftDp(15).children(builder -> builder
                        // 主要消息文本视图，默认显示加载中...
                        .addTextView("message").text("{loading}").textColor(pluginUI.colorTextStateList()).textSize(18)
                        .singleLine().ellipsize(TextUtils.TruncateAt.END) // 单行显示，超出部分用省略号
                        // 次要消息文本视图
                        .addTextView("secondaryMessage").textColor(pluginUI.colorTextSecondaryStateList()).textSize(14)
                        .singleLine().ellipsize(TextUtils.TruncateAt.END).gone() // 初始状态隐藏，有内容时再显示
                )
                .build();

        // 创建不可取消的对话框
        dialog = pluginUI.buildDialog().setView(pluginView).setCancelable(false).create();

        // 获取文本视图引用
        messageView = pluginView.requireViewById("message");
        secondaryMessageView = pluginView.requireViewById("secondaryMessage");

        // 创建UI更新器组，确保线程安全的UI更新
        UIUpdaterGroup updaterGroup = new UIUpdaterGroup();

        // 注册主要消息更新器
        messageUpdater = updaterGroup.registerUpdater(messageView::setText);

        // 注册次要消息更新器，同时控制可见性
        secondaryMessageUpdater = updaterGroup.registerUpdater(text -> {
            // 根据文本内容控制可见性：有内容时显示，无内容时隐藏
            secondaryMessageView.setVisibility(text == null ? PluginView.GONE : PluginView.VISIBLE);
            secondaryMessageView.setText(text);
        });
    }

    @Override
    public PluginDialog getDialog() {
        return dialog;
    }

    /**
     * 设置主要消息文本
     *
     * @param message 要显示的主要消息文本
     * @return 返回当前LoadingDialog实例
     */
    public LoadingDialog setMessage(CharSequence message) {
        messageUpdater.submitUpdate(message);
        return this;
    }

    /**
     * 设置次要消息文本
     *
     * @param secondaryMessage 要显示的次要消息文本，传入null时会隐藏次要消息视图
     * @return 返回当前LoadingDialog实例
     */
    public LoadingDialog setSecondaryMessage(CharSequence secondaryMessage) {
        secondaryMessageUpdater.submitUpdate(secondaryMessage);
        return this;
    }

    /**
     * 立即显示对话框
     * 注意：此方法必须在UI线程中调用
     *
     * @return 返回当前LoadingDialog实例
     */
    public LoadingDialog show() {
        ThreadUtil.assertInUIThread(); // 确保在UI线程中执行
        dialog.show();
        delayShow = false; // 重置延迟显示标志
        return this;
    }

    /**
     * 延迟显示对话框（默认延迟200毫秒）
     *
     * @return 返回当前LoadingDialog实例
     */
    public LoadingDialog showDelay() {
        return showDelay(200);
    }

    /**
     * 延迟指定时间后显示对话框
     *
     * @param delayMillis 延迟时间（毫秒）
     * @return 返回当前LoadingDialog实例
     */
    public LoadingDialog showDelay(long delayMillis) {
        ThreadUtil.assertInUIThread();
        delayShow = true;
        ThreadUtil.postDelayed(new DelayShowRunnable(this), delayMillis);
        return this;
    }

    /**
     * 获取主要消息文本视图
     *
     * @return PluginTextView 主要消息文本视图组件
     */
    public PluginTextView getMessageView() {
        return messageView;
    }

    /**
     * 获取次要消息文本视图
     *
     * @return PluginTextView 次要消息文本视图组件
     */
    public PluginTextView getSecondaryMessageView() {
        return secondaryMessageView;
    }

    /**
     * 检查对话框是否正在显示
     *
     * @return 如果对话框正在显示返回true，否则返回false
     */
    public boolean isShowing() {
        return dialog.isShowing();
    }

    /**
     * 关闭对话框
     * 此方法会自动切换到UI线程执行，确保线程安全
     */
    @Override
    public void dismiss() {
        delayShow = false;
        ThreadUtil.runOnUiThread(() -> {
            if (dialog.isShowing()) {
                dialog.dismiss();
            }
        });
    }

    /**
     * 延迟显示任务
     */
    private static class DelayShowRunnable implements Runnable {
        private final WeakReference<LoadingDialog> ref;

        private DelayShowRunnable(LoadingDialog dialog) {
            this.ref = new WeakReference<>(dialog);
        }

        @Override
        public void run() {
            LoadingDialog loadingDialog = ref.get();
            if (loadingDialog != null && loadingDialog.delayShow) {
                loadingDialog.delayShow = false;
                loadingDialog.dialog.show();
            }
        }
    }
}
