package bin.mt.plugin.api.ui.builder;

import androidx.annotation.Nullable;

import java.util.List;

import bin.mt.json.JSONObject;
import bin.mt.plugin.api.ui.PluginSpinner;

/**
 * 插件下拉选择框构建器接口
 * <p>
 * 下拉选择框是一个让用户从预定义选项列表中选择单个选项的UI组件。
 * 点击下拉框会显示选项列表，用户选择后列表收起并显示所选项。
 * </p>
 * <p>
 * <strong>使用示例：</strong>
 * <pre>{@code
 * List<String> options = Arrays.asList("选项1", "选项2", "选项3");
 * pluginViewBuilder
 *     .addSpinner("language_selector")
 *     .items(options)
 *     .selection(0)
 *     .onItemSelected((spinner, position) -> {
 *         String selected = (String) spinner.getSelectedItem();
 *         // 处理选择事件
 *     })
 *     ...
 * }</pre>
 * </p>
 *
 * @see PluginBaseViewBuilder
 * @see PluginSpinner
 */
public interface PluginSpinnerBuilder extends PluginBaseViewBuilder<PluginSpinnerBuilder> {

    /**
     * 设置下拉框的选项列表
     * <p>
     * 设置用户可以选择的选项数据。列表中的对象如果不是CharSequence类型，
     * 会调用toString()方法转换为字符串进行显示。
     *
     * @param items 选项数据列表，支持任意对象类型
     * @return 当前构建器实例，支持链式调用
     *
     * @see CharSequence
     */
    PluginSpinnerBuilder items(List<?> items);

    /**
     * 设置选中项改变监听器（首次创建完毕后不调用）
     * <p>
     * 使用此方法设置监听器时，在PluginSpinner首次创建完毕后，不会立刻调用监听器传入初始选中状态，
     * 仅在后续选中项发生改变时才会调用
     *
     * @param listener 选择监听器，接收下拉框实例和选中位置索引
     * @return 当前构建器实例，支持链式调用
     */
    PluginSpinnerBuilder postOnItemSelected(PluginSpinner.OnItemSelectedListener listener);

    /**
     * 设置选中项改变监听器（首次创建完毕后立刻调用）
     * <p>
     * 使用此方法设置监听器时，在PluginSpinner首次创建完毕后，立刻调用监听器传入初始选中状态
     *
     * @param listener 选择监听器，接收下拉框实例和选中位置索引
     * @return 当前构建器实例，支持链式调用
     */
    PluginSpinnerBuilder onItemSelected(PluginSpinner.OnItemSelectedListener listener);

    /**
     * 设置当前选中的选项位置
     *
     * @param position 选项位置索引，从0开始
     * @return 当前构建器实例，支持链式调用
     */
    PluginSpinnerBuilder selection(int position);

    /**
     * 从JSON数据中读取选中位置
     * <p>
     * 该方法相当于调用 selection(data.getInt(id, 0))，
     * 后续可通过{@link JSONObject#putSelection(PluginSpinner)}写回数据
     * <p>
     * 注意：调用该方法前必须先通过id()方法设置组件ID。
     *
     * @param data JSON数据源，可以为null
     * @return 当前构建器实例，支持链式调用
     */
    PluginSpinnerBuilder selection(@Nullable JSONObject data);

    /**
     * 从JSON数据中读取选中位置，并指定默认值
     * <p>
     * 该方法相当于调用 selection(data.getInt(id, defaultValue))，
     * 后续可通过{@link JSONObject#putSelection(PluginSpinner)}写回数据
     * <p>
     * 注意：调用该方法前必须先通过id()方法设置组件ID。
     *
     * @param data         JSON数据源，可以为null
     * @param defaultValue 当数据源为null或不包含对应字段时使用的默认选中位置
     * @return 当前构建器实例，支持链式调用
     */
    PluginSpinnerBuilder selection(@Nullable JSONObject data, int defaultValue);
}
