package bin.mt.plugin.api.ui.builder;

import androidx.annotation.NonNull;

import bin.mt.plugin.api.ui.PluginUI;
import bin.mt.plugin.api.util.Consumer;

/**
 * 插件视图组构建器基础接口
 *
 * @param <T> 具体的构建器类型，继承自PluginViewBuilder，用于支持链式调用的类型安全
 * @see PluginBaseViewBuilder
 * @see PluginBaseLinearLayoutBuilder
 */
public interface PluginBaseViewGroupBuilder<T extends PluginViewBuilder> extends PluginBaseViewBuilder<T> {

    /**
     * 获取当前设置的子View样式
     * <p>
     * 返回应用于该PluginViewGroup所有子View的样式配置。如果没有设置自定义样式，
     * 则返回从上一级继承的默认样式。
     * </p>
     *
     * @return 非空的样式对象，用于子View的样式配置
     */
    @NonNull
    PluginUI.Style getChildrenStyle();

    /**
     * 设置子View的统一样式
     * <p>
     * 为该视图组中的所有子View设置统一的样式规则。设置后，通过addTextView()、
     * addButton()等方法添加的子View都会自动应用该样式。
     *
     * @param style 要应用的子View样式，不能为null
     * @return 当前构建器实例，支持链式调用
     */
    T childrenStyle(@NonNull PluginUI.Style style);

    /**
     * 通过Consumer函数式接口来添加子View
     * <p>
     * <strong>使用示例：</strong>
     * <pre>{@code
     * builder.children(childBuilder -> {
     *     childBuilder.addTextView().text("标题").textSize(18);
     *     childBuilder.addButton().text("按钮").onClick(this::onButtonClick);
     *     childBuilder.addEditText().hint("请输入内容");
     * });
     * }</pre>
     * </p>
     *
     * @param consumer 子View构建回调函数，接收子View构建器实例作为参数
     * @return 当前构建器实例，支持链式调用
     */
    T children(Consumer<PluginViewBuilder> consumer);
}
