package bin.mt.plugin.api.ui.builder;

import androidx.annotation.Nullable;

import bin.mt.json.JSONObject;
import bin.mt.plugin.api.ui.PluginCompoundButton;

/**
 * 插件复合按钮构建器基础接口
 * <p>
 * 复合按钮是指具有选中/未选中两种状态的按钮组件，如CheckBox、RadioButton、Switch等。
 * 该接口提供了设置选中状态、监听状态变化等复合按钮特有的功能。
 *
 * @param <T> 具体的构建器类型，使用泛型实现链式调用的类型安全
 * @see PluginBaseButtonBuilder
 * @see PluginCheckBoxBuilder
 * @see PluginSwitchButtonBuilder
 * @see PluginRadioButtonBuilder
 */
public interface PluginBaseCompoundButtonBuilder<T extends PluginViewBuilder> extends PluginBaseButtonBuilder<T> {

    /**
     * 设置选中状态变化监听器
     * <p>
     * 当用户点击复合按钮改变其选中状态时，会触发此监听器回调
     *
     * @param listener 选中状态变化监听器，接收按钮实例和新的选中状态
     * @return 当前构建器实例，支持链式调用
     */
    T onCheckedChange(PluginCompoundButton.OnCheckedChangeListener listener);

    /**
     * 设置按钮的选中状态
     *
     * @param checked true表示选中，false表示未选中
     * @return 当前构建器实例，支持链式调用
     */
    T checked(boolean checked);

    /**
     * 从JSON数据中读取并设置选中状态
     * <p>
     * 该方法相当于调用 checked(data.getBoolean(id, false))，
     * 后续可通过{@link JSONObject#putChecked(PluginCompoundButton)}写回数据
     * <p>
     * 注意：调用该方法前必须先通过id()方法设置组件ID。
     *
     * @param data JSON数据源，可以为null
     * @return 当前构建器实例，支持链式调用
     */
    T checked(@Nullable JSONObject data);

    /**
     * 从JSON数据中读取并设置选中状态，并指定默认值
     * <p>
     * 该方法相当于调用 checked(data.getBoolean(id, defaultValue))，
     * 后续可通过{@link JSONObject#putChecked(PluginCompoundButton)}写回数据
     * <p>
     * 注意：调用该方法前必须先通过id()方法设置组件ID
     *
     * @param data         JSON数据源，可以为null
     * @param defaultValue 当数据源为null或不包含对应字段时使用的默认值
     * @return 当前构建器实例，支持链式调用
     */
    T checked(@Nullable JSONObject data, boolean defaultValue);

    /**
     * 设置按钮为选中状态
     *
     * @return 当前构建器实例，支持链式调用
     */
    T check();

    /**
     * 设置按钮为未选中状态
     *
     * @return 当前构建器实例，支持链式调用
     */
    T uncheck();
}
