package bin.mt.plugin.api.preference;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import bin.mt.plugin.api.LocalString;
import bin.mt.plugin.api.PluginContext;
import bin.mt.plugin.api.ui.PluginUI;

/**
 * 插件设置界面接口
 * <p>
 * 用于构建插件的配置界面，支持多种类型的设置选项，包括文本显示、输入框、开关、单选列表等组件
 */
public interface PluginPreference {

    /**
     * 构建设置界面时的回调方法
     * <p>
     * 在此方法中通过Builder添加各种设置选项
     *
     * @param context MT插件上下文，提供插件运行环境
     * @param builder 界面构建器，用于添加设置选项
     */
    void onBuild(PluginContext context, Builder builder);

    /**
     * 设置界面构建器
     * <p>
     * 用于添加和配置各种设置选项组件
     */
    interface Builder {

        /**
         * 设置界面顶栏的标题，默认为"设置"
         * <p>
         * 如果title类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本
         * <p>
         * 在一些特殊情况下若不想被转化为本地化文本，可以传入非String类型，例如：
         * <code>title(new SpannedString("{key}"))</code>
         * <p>
         * 关于key的详细介绍请参考 {@link PluginContext#getString(String)}
         *
         * @param title 标题文本
         * @return Builder实例，可进行进一步配置
         * @see PluginContext#getString(String)
         */
        Builder title(@NonNull CharSequence title);

        /**
         * 设置界面顶栏的副标题，默认为空
         * <p>
         * 支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
         *
         * @param subtitle 副标题文本
         * @return Builder实例，可进行进一步配置
         */
        Builder subtitle(@NonNull CharSequence subtitle);

        /**
         * 添加一个分组标题头
         * <p>
         * 用于将相关的设置选项进行分组显示
         * <p>
         * 支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
         *
         * @param title 分组标题文本
         */
        void addHeader(@NonNull CharSequence title);

        /**
         * 添加一个纯文本显示项
         * <p>
         * 可用于显示说明信息或链接，也可以自定义点击事件
         * <p>
         * 支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
         *
         * @param title 显示的标题文本
         * @return Text实例，可进行进一步配置
         */
        Text addText(@NonNull CharSequence title);

        /**
         * 添加一个文本输入选项
         * <p>
         * 用户可以输入文本内容，值以String类型存储
         * <p>
         * title支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
         *
         * @param title 选项标题
         * @param key   存储选项值的键名，用于SharedPreferences
         * @return Input实例，可进行进一步配置
         */
        Input addInput(@NonNull CharSequence title, @NonNull String key);

        /**
         * 添加一个开关选项
         * <p>
         * 用户可以开启或关闭，值以boolean类型存储
         * <p>
         * title支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
         *
         * @param title 选项标题
         * @param key   存储选项值的键名，用于SharedPreferences
         * @return Switch实例，可进行进一步配置
         */
        Switch addSwitch(@NonNull CharSequence title, @NonNull String key);

        /**
         * 添加一个单选列表选项
         * <p>
         * 用户可以从预定义的选项中选择一个，值以String类型存储
         * <p>
         * title支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
         *
         * @param title 选项标题
         * @param key   存储选项值的键名，用于SharedPreferences
         * @return List实例，可进行进一步配置
         */
        List addList(@NonNull CharSequence title, @NonNull String key);

        /**
         * 设置本地化文本实例
         *
         * @param localString 本地化文本实例
         * @deprecated 仅为兼容v2的API保留，请不要再使用该方法
         */
        @Deprecated
        void setLocalString(LocalString localString);

    }

    /**
     * 纯文本显示选项
     * <p>
     * 用于显示静态文本信息或可点击的链接，也可以自定义点击事件
     */
    interface Text {

        /**
         * 设置摘要文本
         * <p>
         * 显示在标题下方的详细说明信息
         * <p>
         * 支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
         *
         * @param summary 摘要文本内容
         * @return 当前Text实例，支持链式调用
         */
        Text summary(CharSequence summary);

        /**
         * 设置点击后跳转的URL地址
         * <p>
         * <strong>注意：</strong>如果同时设置了onClick监听器，
         * 自定义点击事件的优先级更高，URL跳转将不会执行。
         *
         * @param url 要跳转的URL地址，应为有效的HTTP/HTTPS链接
         * @return 当前Text实例，支持链式调用
         * @see #onClick(OnTextItemClickListener)
         */
        Text url(String url);

        /**
         * 设置自定义点击事件监听器
         * <p>
         * 当用户点击该文本选项时，会触发自定义的点击事件处理逻辑。
         * 如果设置了此监听器，将覆盖通过{@link #url(String)}设置的URL跳转行为。
         *
         * @param listener 点击事件监听器，不能为null
         * @return 当前Text实例，支持链式调用
         * @see OnTextItemClickListener
         * @see #url(String)
         */
        Text onClick(OnTextItemClickListener listener);

    }

    /**
     * 文本选项点击事件监听器接口
     */
    interface OnTextItemClickListener {

        /**
         * 处理文本选项点击事件
         *
         * @param pluginUI PluginUI实例实例，可以用于显示自定义对话框
         * @param item     被点击的文本项实例，包含标题、摘要等信息
         */
        void onClick(PluginUI pluginUI, TextItem item);
    }

    /**
     * 文本选项数据项接口
     */
    interface TextItem {

        /**
         * 获取文本选项的标题
         *
         * @return 标题文本
         */
        @NonNull
        CharSequence getTitle();

        /**
         * 设置文本选项的标题
         * <p>
         * 支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
         *
         * @param title 新的标题文本
         */
        void setTitle(@Nullable CharSequence title);

        /**
         * 获取文本选项的摘要信息
         *
         * @return 摘要文本
         */
        @NonNull
        CharSequence getSummary();

        /**
         * 设置文本选项的摘要信息
         * <p>
         * 支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
         *
         * @param summary 新的摘要文本
         */
        void setSummary(@Nullable CharSequence summary);
    }

    /**
     * 文本输入选项
     * 允许用户输入和编辑文本内容
     */
    interface Input {

        /**
         * 设置输入框的默认值
         * <p>
         * 支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
         *
         * @param defaultValue 默认文本内容
         * @return 当前Input实例，支持链式调用
         */
        Input defaultValue(CharSequence defaultValue);

        /**
         * 设置摘要文本
         * <p>
         * 显示在标题下方的说明信息
         * <p>
         * 支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
         *
         * @param summary 摘要文本内容
         * @return 当前Input实例，支持链式调用
         */
        Input summary(CharSequence summary);

        /**
         * 使用当前输入值作为摘要显示
         * <p>
         * 此选项会覆盖通过summary()方法设置的摘要
         *
         * @return 当前Input实例，支持链式调用
         */
        Input valueAsSummary();

        /**
         * 设置文本输入类型
         * <p>
         * 用于控制软键盘类型和输入限制
         *
         * @param inputType 输入类型，参考{@link android.text.InputType}常量
         * @return 当前Input实例，支持链式调用
         */
        Input inputType(int inputType);

        /**
         * 设置输入内容校验器
         * <p>
         * 用于检测用户输入内容的格式是否正确
         *
         * @param validator 校验器实例
         * @return 当前Input实例，支持链式调用
         */
        Input validator(Validator validator);

        /**
         * 输入内容校验器接口
         * <p>
         * 用于验证用户输入的内容是否符合要求
         */
        interface Validator {

            /**
             * 校验输入内容
             * <p>
             * 返回值支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
             *
             * @param value 待校验的输入内容
             * @return 校验通过返回null，否则返回错误信息
             * 错误信息支持本地化文本{key}格式
             */
            @Nullable
            String validate(String value);
        }
    }

    /**
     * 开关选项
     * 提供开启/关闭两种状态的选择
     */
    interface Switch {

        /**
         * 设置开关的默认状态
         *
         * @param defaultValue true为开启，false为关闭
         * @return 当前Switch实例，支持链式调用
         */
        Switch defaultValue(boolean defaultValue);

        /**
         * 设置摘要文本
         * <p>
         * 显示在标题下方的说明信息
         * <p>
         * 支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
         *
         * @param summary 摘要文本内容
         * @return 当前Switch实例，支持链式调用
         */
        Switch summary(CharSequence summary);

        /**
         * 设置开关为"开启"状态时显示的摘要
         * <p>
         * 此选项会覆盖通过summary()方法设置的摘要
         * <p>
         * 支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
         *
         * @param summary 开启状态的摘要文本
         * @return 当前Switch实例，支持链式调用
         */
        Switch summaryOn(CharSequence summary);

        /**
         * 设置开关为"关闭"状态时显示的摘要
         * <p>
         * 此选项会覆盖通过summary()方法设置的摘要
         * <p>
         * 支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
         *
         * @param summary 关闭状态的摘要文本
         * @return 当前Switch实例，支持链式调用
         */
        Switch summaryOff(CharSequence summary);
    }

    /**
     * 单选列表选项
     * 允许用户从预定义的选项中选择一个
     */
    interface List {

        /**
         * 设置列表的默认选中值
         * <p>
         * 支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
         *
         * @param defaultValue 默认选中项的值
         * @return 当前List实例，支持链式调用
         */
        List defaultValue(CharSequence defaultValue);

        /**
         * 设置摘要文本
         * <p>
         * 显示在标题下方的说明信息
         * <p>
         * 支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
         *
         * @param summary 摘要文本内容
         * @return 当前List实例，支持链式调用
         */
        List summary(CharSequence summary);

        /**
         * 添加一个子选项到列表中
         * <p>
         * name支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
         *
         * @param name  子选项显示名称
         * @param value 子选项对应的值
         * @return Item实例，可进行进一步配置或添加更多选项
         */
        Item addItem(@NonNull CharSequence name, String value);

        /**
         * 单选列表中的子选项
         */
        interface Item {

            /**
             * 设置该子选项被选中时显示的摘要
             * <p>
             * 此选项会覆盖父列表通过summary()方法设置的摘要
             * <p>
             * 支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
             *
             * @param summary 选中时的摘要文本
             * @return 当前Item实例，支持链式调用
             */
            Item summary(CharSequence summary);

            /**
             * 添加下一个子选项到列表中
             * <p>
             * name支持<code>{key}</code>格式引用本地化文本，具体请参考{@link Builder#title(CharSequence)}
             *
             * @param name  子选项显示名称
             * @param value 子选项对应的值
             * @return 新的Item实例，可进行进一步配置
             */
            Item addItem(@NonNull CharSequence name, String value);
        }
    }
}
