package bin.mt.plugin.api.editor;

import android.text.GetChars;

import androidx.annotation.NonNull;

import bin.mt.plugin.api.regex.Matcher;
import bin.mt.plugin.api.regex.Pattern;

/**
 * 文本编辑器的缓存文本类，可高效地进行字符获取与文本查找操作。
 * <p>
 * 该接口继承了 CharSequence 和 GetChars，提供了基础的字符序列操作能力，
 * 同时扩展了高效的文本搜索、行定位和正则匹配功能。
 * </p>
 * <strong>关于换行符</strong>
 * <p>
 * MT 将文本读取到内存中时总是会将换行符强制转换为<code>\n</code>，写出文件时再转回目标换行符。
 * 因此在 BufferedText 中换行符永远都是 <code>\n</code>，不会是 <code>\r</code> 或者 <code>\r\n</code>。
 * </p>
 */
public interface BufferedText extends CharSequence, GetChars {

    /**
     * 从文本开头查找指定字符出现的位置
     *
     * @param c 要查找的字符
     * @return 字符首次出现的位置索引，如果未找到则返回 -1
     */
    int indexOf(char c);

    /**
     * 从指定位置开始查找指定字符出现的位置
     *
     * @param c         要查找的字符
     * @param fromIndex 开始查找的位置索引（包含该位置）
     * @return 字符首次出现的位置索引，如果未找到则返回 -1
     * @throws IndexOutOfBoundsException 如果 fromIndex 小于 0 或大于文本长度
     */
    int indexOf(char c, int fromIndex);

    /**
     * 从文本开头查找指定字符串出现的位置
     *
     * @param s 要查找的字符串，不能为 null
     * @return 字符串首次出现的位置索引，如果未找到则返回 -1
     * @throws NullPointerException 如果 s 为 null
     */
    int indexOf(String s);

    /**
     * 从指定位置开始查找指定字符串出现的位置
     *
     * @param s         要查找的字符串，不能为 null
     * @param fromIndex 开始查找的位置索引（包含该位置）
     * @return 字符串首次出现的位置索引，如果未找到则返回 -1
     * @throws NullPointerException      如果 s 为 null
     * @throws IndexOutOfBoundsException 如果 fromIndex 小于 0 或大于文本长度
     */
    int indexOf(String s, int fromIndex);

    /**
     * 从文本末尾开始倒序查找指定字符出现的位置
     *
     * @param c 要查找的字符
     * @return 字符最后一次出现的位置索引，如果未找到则返回 -1
     */
    int lastIndexOf(char c);

    /**
     * 从指定位置开始倒序查找指定字符出现的位置
     *
     * @param c         要查找的字符
     * @param fromIndex 开始倒序查找的位置索引（包含该位置）
     * @return 字符最后一次出现的位置索引，如果未找到则返回 -1
     * @throws IndexOutOfBoundsException 如果 fromIndex 小于 0 或大于文本长度
     */
    int lastIndexOf(char c, int fromIndex);

    /**
     * 从文本末尾开始倒序查找指定字符串出现的位置
     *
     * @param s 要查找的字符串，不能为 null
     * @return 字符串最后一次出现的位置索引，如果未找到则返回 -1
     * @throws NullPointerException 如果 s 为 null
     */
    int lastIndexOf(String s);

    /**
     * 从指定位置开始倒序查找指定字符串出现的位置
     *
     * @param s         要查找的字符串，不能为 null
     * @param fromIndex 开始倒序查找的位置索引（包含该位置）
     * @return 字符串最后一次出现的位置索引，如果未找到则返回 -1
     * @throws NullPointerException      如果 s 为 null
     * @throws IndexOutOfBoundsException 如果 fromIndex 小于 0 或大于文本长度
     */
    int lastIndexOf(String s, int fromIndex);

    /**
     * 重置缓存数据
     * <p>
     * 当编辑器内的文本被修改后，由于缓存机制可能导致无法获取到最新的内容，
     * 调用此方法可以清空所有缓存，确保后续操作能够获取到最新的文本数据。
     * </p>
     * <p>
     * 注意：调用此方法后，之前缓存的数据将被清除，下次访问时会重新构建缓存，
     * 可能会有一定的性能开销。
     * </p>
     */
    void reset();

    /**
     * 统计指定范围内某个字符的出现次数
     *
     * @param fromIndex 起始位置索引（包含）
     * @param toIndex   结束位置索引（不包含）
     * @param c         要统计的字符
     * @return 字符在指定范围内的出现次数
     * @throws IndexOutOfBoundsException 如果 fromIndex < 0 或 toIndex > length() 或 fromIndex > toIndex
     */
    int count(int fromIndex, int toIndex, char c);

    /**
     * 从指定位置向前查找所在行的开头位置
     * <p>
     * 行开头定义为文本开始位置或换行符（\n）的下一个位置。
     * </p>
     *
     * @param fromIndex 开始查找的位置索引
     * @return 所在行的开头位置索引
     * @throws IndexOutOfBoundsException 如果 fromIndex 小于 0 或大于等于文本长度
     */
    int findLineStart(int fromIndex);

    /**
     * 从指定位置向后查找所在行的结尾位置
     * <p>
     * 行结尾定义为换行符（\n）的位置或文本的末尾位置。
     * </p>
     *
     * @param fromIndex 开始查找的位置索引
     * @return 所在行的结尾位置索引
     * @throws IndexOutOfBoundsException 如果 fromIndex 小于 0 或大于等于文本长度
     */
    int findLineEnd(int fromIndex);

    /**
     * 获取一个基于指定正则表达式模式的匹配器
     * <p>
     * 不要使用系统库的 {@link java.util.regex.Matcher}，其会调用传入的 text 的 toString() 方法，
     * 对于大文本，这不但低效，还浪费内存。
     * </p>
     *
     * @param pattern 正则表达式模式，不能为 null
     * @return 正则表达式匹配器实例
     * @throws NullPointerException 如果 pattern 为 null
     * @see Matcher
     * @see Pattern
     */
    Matcher matcher(Pattern pattern);

    /**
     * 获取一个基于指定正则表达式模式的匹配器
     * <p>
     * 不要使用系统库的 {@link java.util.regex.Matcher}，其会调用传入的 text 的 toString() 方法，
     * 对于大文本，这不但低效，还浪费内存。
     * </p>
     *
     * @param pattern 正则表达式模式，不能为 null
     * @return 正则表达式匹配器实例
     * @throws NullPointerException 如果 pattern 为 null
     * @see Matcher
     * @see Pattern
     */
    Matcher matcher(String pattern);

    /**
     * @deprecated 尽量避免调用此方法，对于超大文本，此方法会生成一个超大字符串对象，可能导致内存不足
     */
    @NonNull
    @Override
    @Deprecated
    String toString();
}
