/*******************************************************************************
 * Copyright (c) 2015, 2016 EclipseSource.
 * <p>
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * <p>
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * <p>
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 ******************************************************************************/
package bin.mt.json;

import java.io.IOException;
import java.io.Reader;


/**
 * This class serves as the entry point to the minimal-json API.
 * <p>
 * To <strong>parse</strong> a given JSON input, use the <code>parse()</code> methods like in this
 * example:
 * </p>
 * <pre>
 * JSONObject object = JSON.parse(string).asObject();
 * </pre>
 * <p>
 * To <strong>create</strong> a JSON data structure to be serialized, use the methods
 * <code>value()</code>, <code>array()</code>, and <code>object()</code>. For example, the following
 * snippet will produce the JSON string <em>{"foo": 23, "bar": true}</em>:
 * </p>
 * <pre>
 * String string = JSON.object().add("foo", 23).add("bar", true).toString();
 * </pre>
 * <p>
 * To create a JSON array from a given Java array, you can use one of the <code>array()</code>
 * methods with varargs parameters:
 * </p>
 * <pre>
 * String[] names = ...
 * JSONArray array = JSON.array(names);
 * </pre>
 */
public final class JSON {

    private JSON() {
        // not meant to be instantiated
    }

    /**
     * Represents the JSON literal <code>null</code>.
     */
    public static final JSONValue NULL = new JSONLiteral("null");

    /**
     * Represents the JSON literal <code>true</code>.
     */
    public static final JSONValue TRUE = new JSONLiteral("true");

    /**
     * Represents the JSON literal <code>false</code>.
     */
    public static final JSONValue FALSE = new JSONLiteral("false");

    /**
     * Returns a JSONValue instance that represents the given <code>int</code> value.
     *
     * @param value the value to get a JSON representation for
     * @return a JSON value that represents the given value
     */
    public static JSONValue value(int value) {
        return new JSONNumber(Integer.toString(value, 10));
    }

    /**
     * Returns a JSONValue instance that represents the given <code>long</code> value.
     *
     * @param value the value to get a JSON representation for
     * @return a JSON value that represents the given value
     */
    public static JSONValue value(long value) {
        return new JSONNumber(Long.toString(value, 10));
    }

    /**
     * Returns a JSONValue instance that represents the given <code>float</code> value.
     *
     * @param value the value to get a JSON representation for
     * @return a JSON value that represents the given value
     */
    public static JSONValue value(float value) {
        if (Float.isInfinite(value) || Float.isNaN(value)) {
            throw new IllegalArgumentException("Infinite and NaN values not permitted in JSON");
        }
        return new JSONNumber(cutOffPointZero(Float.toString(value)));
    }

    /**
     * Returns a JSONValue instance that represents the given <code>double</code> value.
     *
     * @param value the value to get a JSON representation for
     * @return a JSON value that represents the given value
     */
    public static JSONValue value(double value) {
        if (Double.isInfinite(value) || Double.isNaN(value)) {
            throw new IllegalArgumentException("Infinite and NaN values not permitted in JSON");
        }
        return new JSONNumber(cutOffPointZero(Double.toString(value)));
    }

    /**
     * Returns a JSONValue instance that represents the given string.
     *
     * @param string the string to get a JSON representation for
     * @return a JSON value that represents the given string
     */
    public static JSONValue value(String string) {
        return string == null ? NULL : new JSONString(string);
    }

    /**
     * Returns a JSONValue instance that represents the given <code>boolean</code> value.
     *
     * @param value the value to get a JSON representation for
     * @return a JSON value that represents the given value
     */
    public static JSONValue value(boolean value) {
        return value ? TRUE : FALSE;
    }

    /**
     * Creates a new empty JSONArray. This is equivalent to creating a new JSONArray using the
     * constructor.
     *
     * @return a new empty JSON array
     */
    public static JSONArray array() {
        return new JSONArray();
    }

    /**
     * Creates a new JSONArray that contains the JSON representations of the given <code>int</code>
     * values.
     *
     * @param values the values to be included in the new JSON array
     * @return a new JSON array that contains the given values
     */
    public static JSONArray array(int... values) {
        if (values == null) {
            throw new NullPointerException("values is null");
        }
        JSONArray array = new JSONArray();
        for (int value : values) {
            array.add(value);
        }
        return array;
    }

    /**
     * Creates a new JSONArray that contains the JSON representations of the given <code>long</code>
     * values.
     *
     * @param values the values to be included in the new JSON array
     * @return a new JSON array that contains the given values
     */
    public static JSONArray array(long... values) {
        if (values == null) {
            throw new NullPointerException("values is null");
        }
        JSONArray array = new JSONArray();
        for (long value : values) {
            array.add(value);
        }
        return array;
    }

    /**
     * Creates a new JSONArray that contains the JSON representations of the given <code>float</code>
     * values.
     *
     * @param values the values to be included in the new JSON array
     * @return a new JSON array that contains the given values
     */
    public static JSONArray array(float... values) {
        if (values == null) {
            throw new NullPointerException("values is null");
        }
        JSONArray array = new JSONArray();
        for (float value : values) {
            array.add(value);
        }
        return array;
    }

    /**
     * Creates a new JSONArray that contains the JSON representations of the given <code>double</code>
     * values.
     *
     * @param values the values to be included in the new JSON array
     * @return a new JSON array that contains the given values
     */
    public static JSONArray array(double... values) {
        if (values == null) {
            throw new NullPointerException("values is null");
        }
        JSONArray array = new JSONArray();
        for (double value : values) {
            array.add(value);
        }
        return array;
    }

    /**
     * Creates a new JSONArray that contains the JSON representations of the given
     * <code>boolean</code> values.
     *
     * @param values the values to be included in the new JSON array
     * @return a new JSON array that contains the given values
     */
    public static JSONArray array(boolean... values) {
        if (values == null) {
            throw new NullPointerException("values is null");
        }
        JSONArray array = new JSONArray();
        for (boolean value : values) {
            array.add(value);
        }
        return array;
    }

    /**
     * Creates a new JSONArray that contains the JSON representations of the given strings.
     *
     * @param strings the strings to be included in the new JSON array
     * @return a new JSON array that contains the given strings
     */
    public static JSONArray array(String... strings) {
        if (strings == null) {
            throw new NullPointerException("values is null");
        }
        JSONArray array = new JSONArray();
        for (String value : strings) {
            array.add(value);
        }
        return array;
    }

    /**
     * Creates a new empty JSONObject. This is equivalent to creating a new JSONObject using the
     * constructor.
     *
     * @return a new empty JSON object
     */
    public static JSONObject object() {
        return new JSONObject();
    }

    /**
     * Parses the given input string as JSON. The input must contain a valid JSON value, optionally
     * padded with whitespace.
     *
     * @param source the input string, must be valid JSON
     * @return a value that represents the parsed JSON
     * @throws JSONParseException if the input is not valid JSON
     */
    public static JSONValue parse(String source) {
        if (source == null) {
            throw new NullPointerException("string is null");
        }
        DefaultHandler handler = new DefaultHandler();
        new JSONParser(handler).parse(source);
        return handler.getValue();
    }

    /**
     * Reads the entire input from the given reader and parses it as JSON. The input must contain a
     * valid JSON value, optionally padded with whitespace.
     * <p>
     * Characters are read in chunks into an input buffer. Hence, wrapping a reader in an additional
     * <code>BufferedReader</code> likely won't improve reading performance.
     * </p>
     *
     * @param reader the reader to read the JSON value from
     * @return a value that represents the parsed JSON
     * @throws IOException        if an I/O error occurs in the reader
     * @throws JSONParseException if the input is not valid JSON
     */
    public static JSONValue parse(Reader reader) throws IOException {
        if (reader == null) {
            throw new NullPointerException("reader is null");
        }
        DefaultHandler handler = new DefaultHandler();
        new JSONParser(handler).parse(reader);
        return handler.getValue();
    }

    private static String cutOffPointZero(String string) {
        if (string.endsWith(".0")) {
            return string.substring(0, string.length() - 2);
        }
        return string;
    }

    static class DefaultHandler extends JSONHandler<JSONArray, JSONObject> {

        protected JSONValue value;

        @Override
        public JSONArray startArray() {
            return new JSONArray();
        }

        @Override
        public JSONObject startObject() {
            return new JSONObject();
        }

        @Override
        public void endNull() {
            value = NULL;
        }

        @Override
        public void endBoolean(boolean bool) {
            value = bool ? TRUE : FALSE;
        }

        @Override
        public void endString(String string) {
            value = new JSONString(string);
        }

        @Override
        public void endNumber(String string) {
            value = new JSONNumber(string);
        }

        @Override
        public void endArray(JSONArray array) {
            value = array;
        }

        @Override
        public void endObject(JSONObject object) {
            value = object;
        }

        @Override
        public void endArrayValue(JSONArray array) {
            array.add(value);
        }

        @Override
        public void endObjectValue(JSONObject object, String name) {
            object.add(name, value);
        }

        JSONValue getValue() {
            return value;
        }

    }

}
